<?php


class SSHCommand
{
    protected $_ssh_info = null;
    protected $_connection = null;

    function __construct(SSHInfo $ssh_info)
    {
        $this->_ssh_info = $ssh_info;
        $this->_connection = ssh2_connect($ssh_info->getHost(), $ssh_info->getPort());
        $ret = ssh2_auth_password($this->_connection, $ssh_info->getUser(), $ssh_info->getPwd());
        if (!$ret) {
            throw new Exception("SSH authentication fail, host: " . var_export($ssh_info, true));
        }
    }

    public function doCmdAndCheckExitCode($cmd, $blocking = true)
    {
        $get_exit_code_cmd = $cmd . ';echo $?';
        $out = $this->doCmd($get_exit_code_cmd, $blocking);
        $trim_out = rtrim($out);
        $exit_code = (int)substr($trim_out, -1);
        $out_without_exit_code = substr($trim_out, 0, -1);
        if($exit_code != 0) throw new Exception("Do ssh command fail, $cmd, output is $out_without_exit_code, exit code is $exit_code");
        return $out_without_exit_code;
    }

    //Blocking means a read from the *stream* will wait until there is data.
    public function doCmd($cmd, $blocking = true)
    {
        $stream = ssh2_exec($this->_connection, $cmd);
        if ($stream == false) throw new Exception("ssh execute command fail. host: " . var_export($this->_ssh_info, true));
        stream_set_blocking($stream, $blocking);
        $out = stream_get_contents($stream);
        fclose($stream);
        return $out;
    }
}

class SSHInfo
{
    protected $_host = null;
    protected $_user = null;
    protected $_pwd = null;
    protected $_port = null;

    function __construct($host, $user, $pwd, $port = 22)
    {
        $this->_host = $host;
        $this->_user = $user;
        $this->_pwd = $pwd;
        $this->_port = $port;
    }

    public function setHost($host)
    {
        $this->_host = $host;
    }

    public function getHost()
    {
        return $this->_host;
    }

    public function setPort($port)
    {
        $this->_port = $port;
    }

    public function getPort()
    {
        return $this->_port;
    }

    public function setPwd($pwd)
    {
        $this->_pwd = $pwd;
    }

    public function getPwd()
    {
        return $this->_pwd;
    }

    public function setUser($user)
    {
        $this->_user = $user;
    }

    public function getUser()
    {
        return $this->_user;
    }
}

abstract class ClusterSSHServerCmdRetriever extends SSHServerCmdRetriever
{
    protected $_sc_id = null;

    function __construct($cfg_xml_path, $sc_id)
    {
        $this->_sc_id = $sc_id;
        parent::__construct($cfg_xml_path);
    }
}

abstract class SSHServerCmdRetriever
{
    protected $_cfg_xml = null;
    protected $_cfg_xml_path = null;

    function __construct($cfg_xml_path)
    {
        $this->_cfg_xml_path = $cfg_xml_path;
        $this->_cfg_xml = simplexml_load_file($cfg_xml_path);
    }

    abstract protected function getSSHInfo();

    public function getSSHCmd()
    {
        $ssh_info = $this->getSSHInfo();
        return new SSHCommand($ssh_info);
    }
}

class ClusterManagementServerSSHCmdRetriever extends ClusterSSHServerCmdRetriever
{
    protected function getSSHInfo()
    {
        $cluster_section = "cluster_$this->_sc_id";
        $ip = (string)$this->_cfg_xml->$cluster_section->management_server->ip;
        $user = (string)$this->_cfg_xml->$cluster_section->management_server->admin_username;
        $pwd = (string)$this->_cfg_xml->$cluster_section->management_server->admin_password;
        return new SSHInfo($ip, $user, $pwd);
    }
}

class ClusterSandboxControllerSSHCmdRetriever extends ClusterSSHServerCmdRetriever
{
    protected function getSSHInfo()
    {
        $cluster_section = "cluster_$this->_sc_id";
        $ip = (string)$this->_cfg_xml->$cluster_section->management_server->ip;
        $user = (string)$this->_cfg_xml->$cluster_section->sandbox_controller->admin_username;
        $pwd = (string)$this->_cfg_xml->$cluster_section->sandbox_controller->admin_password;
        return new SSHInfo($ip, $user, $pwd, 1122);
    }
}

class ManagementServerSSHCmdRetriever extends SSHServerCmdRetriever
{
    protected function getSSHInfo()
    {
        $ip = (string)$this->_cfg_xml->management_server->ip;
        $user = (string)$this->_cfg_xml->management_server->admin_username;
        $pwd = (string)$this->_cfg_xml->management_server->admin_password;
        return new SSHInfo($ip, $user, $pwd);
    }
}

class InternalSandboxControllerSSHCmdRetriever extends SSHServerCmdRetriever
{
    protected function getSSHInfo()
    {
        $ip = (string)$this->_cfg_xml->sandbox_controller->ip;
        $user = (string)$this->_cfg_xml->sandbox_controller->admin_username;
        $pwd = (string)$this->_cfg_xml->sandbox_controller->admin_password;
        return new SSHInfo($ip, $user, $pwd);
    }
}

class SandboxControllerSSHCmdRetriever extends SSHServerCmdRetriever
{
    protected function getSSHInfo()
    {
        $ip = (string)$this->_cfg_xml->management_server->ip;
        $port = 1122;
        $user = (string)$this->_cfg_xml->sandbox_controller->admin_username;
        $pwd = (string)$this->_cfg_xml->sandbox_controller->admin_password;
        return new SSHInfo($ip, $user, $pwd, $port);
    }
}

